/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.accumulo.classloader.ccl;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.ResourceHandler;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

public class TestUtils {

  public static class TestClassInfo {
    private final String className;
    private final String helloOutput;

    public TestClassInfo(String className, String helloOutput) {
      super();
      this.className = className;
      this.helloOutput = helloOutput;
    }

    public String getClassName() {
      return className;
    }

    public String getHelloOutput() {
      return helloOutput;
    }
  }

  public static org.apache.hadoop.fs.Path createManifestFile(FileSystem fs, String name,
      String contents) throws Exception {
    var baseHdfsPath = new org.apache.hadoop.fs.Path("/remote-manifests");
    assertTrue(fs.mkdirs(baseHdfsPath));
    var newFile = new org.apache.hadoop.fs.Path(baseHdfsPath, name);

    if (contents == null) {
      assertTrue(fs.createNewFile(newFile));
    } else {
      try (FSDataOutputStream out = fs.create(newFile)) {
        out.writeBytes(contents);
      }
    }
    assertTrue(fs.exists(newFile));
    return newFile;
  }

  public static void updateManifestFile(FileSystem fs, org.apache.hadoop.fs.Path path,
      String contents) throws Exception {
    // Update the contents
    assertTrue(fs.exists(path));
    fs.delete(path, false);
    assertFalse(fs.exists(path));

    if (contents == null) {
      assertTrue(fs.createNewFile(path));
    } else {
      try (FSDataOutputStream out = fs.create(path)) {
        out.writeBytes(contents);
      }
    }
    assertTrue(fs.exists(path));

  }

  public static void testClassLoads(ClassLoader cl, TestClassInfo tci) throws Exception {
    var clazz = cl.loadClass(tci.getClassName()).asSubclass(test.Test.class);
    test.Test impl = clazz.getDeclaredConstructor().newInstance();
    assertEquals(tci.getHelloOutput(), impl.hello());
  }

  public static void testClassFailsToLoad(ClassLoader cl, TestClassInfo tci) throws Exception {
    assertThrows(ClassNotFoundException.class, () -> cl.loadClass(tci.getClassName()));
  }

  public static MiniDFSCluster getMiniCluster() throws IOException {
    System.setProperty("java.io.tmpdir", System.getProperty("user.dir") + "/target");

    // Put the MiniDFSCluster directory in the target directory
    System.setProperty("test.build.data", "target/build/test/data");

    // Setup HDFS
    Configuration conf = new Configuration();
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, 1024 * 1024); // 1M blocksize

    MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf).build();
    cluster.waitClusterUp();
    return cluster;
  }

  public static Server getJetty(Path resourceDirectory) throws Exception {
    ResourceHandler handler = new ResourceHandler();
    // work with jetty 11 or 12, so we can build with Accumulo 2.1 or 4.0
    Method m;
    try {
      m = handler.getClass().getMethod("setResourceBase", String.class); // jetty 11
    } catch (NoSuchMethodException e) {
      m = handler.getClass().getMethod("setBaseResourceAsString", String.class); // jetty 12
    }
    m.invoke(handler, resourceDirectory.toString());

    Server jetty = new Server(0);
    jetty.setHandler(handler);
    jetty.start();
    return jetty;
  }

  @SuppressFBWarnings(value = "URLCONNECTION_SSRF_FD",
      justification = "user-supplied URL is the intended functionality")
  public static String computeResourceChecksum(String algorithm, URL resourceLocation)
      throws IOException {
    try (InputStream is = resourceLocation.openStream()) {
      return new DigestUtils(algorithm).digestAsHex(is);
    }
  }

  @SuppressFBWarnings(value = "URLCONNECTION_SSRF_FD",
      justification = "user-supplied URL is the intended functionality")
  public static long getFileSize(URL url) throws IOException {
    try (InputStream is = url.openStream()) {
      return IOUtils.consume(is);
    }
  }

  public static long getFileSize(Path p) throws IOException {
    try (InputStream is = Files.newInputStream(p)) {
      return IOUtils.consume(is);
    }
  }

}
