/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hadoopbackport;

import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.fs.PositionedReadable;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.yetus.audience.InterfaceAudience;

@InterfaceAudience.Private
public class ThrottledInputStream
extends InputStream {
    private final InputStream rawStream;
    private final long maxBytesPerSec;
    private final long startTime = EnvironmentEdgeManager.currentTime();
    private long bytesRead = 0L;
    private long totalSleepTime = 0L;

    public ThrottledInputStream(InputStream rawStream) {
        this(rawStream, Long.MAX_VALUE);
    }

    public ThrottledInputStream(InputStream rawStream, long maxBytesPerSec) {
        assert (maxBytesPerSec > 0L) : "Bandwidth " + maxBytesPerSec + " is invalid";
        this.rawStream = rawStream;
        this.maxBytesPerSec = maxBytesPerSec;
    }

    @Override
    public void close() throws IOException {
        this.rawStream.close();
    }

    @Override
    public int read() throws IOException {
        this.throttle();
        int data = this.rawStream.read();
        if (data != -1) {
            ++this.bytesRead;
        }
        return data;
    }

    @Override
    public int read(byte[] b) throws IOException {
        this.throttle();
        int readLen = this.rawStream.read(b);
        if (readLen != -1) {
            this.bytesRead += (long)readLen;
        }
        return readLen;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.throttle();
        int readLen = this.rawStream.read(b, off, len);
        if (readLen != -1) {
            this.bytesRead += (long)readLen;
        }
        return readLen;
    }

    public int read(long position, byte[] buffer, int offset, int length) throws IOException {
        if (!(this.rawStream instanceof PositionedReadable)) {
            throw new UnsupportedOperationException("positioned read is not supported by the internal stream");
        }
        this.throttle();
        int readLen = ((PositionedReadable)this.rawStream).read(position, buffer, offset, length);
        if (readLen != -1) {
            this.bytesRead += (long)readLen;
        }
        return readLen;
    }

    private long calSleepTimeMs() {
        return ThrottledInputStream.calSleepTimeMs(this.bytesRead, this.maxBytesPerSec, EnvironmentEdgeManager.currentTime() - this.startTime);
    }

    static long calSleepTimeMs(long bytesRead, long maxBytesPerSec, long elapsed) {
        assert (elapsed > 0L) : "The elapsed time should be greater than zero";
        if (bytesRead <= 0L || maxBytesPerSec <= 0L) {
            return 0L;
        }
        long rval = (long)((double)bytesRead / (double)maxBytesPerSec * 1000.0 - (double)elapsed);
        if (rval <= 0L) {
            return 0L;
        }
        return rval;
    }

    private void throttle() throws InterruptedIOException {
        long sleepTime = this.calSleepTimeMs();
        this.totalSleepTime += sleepTime;
        try {
            TimeUnit.MILLISECONDS.sleep(sleepTime);
        }
        catch (InterruptedException e) {
            throw new InterruptedIOException("Thread aborted");
        }
    }

    public long getTotalBytesRead() {
        return this.bytesRead;
    }

    public long getBytesPerSec() {
        long elapsed = (EnvironmentEdgeManager.currentTime() - this.startTime) / 1000L;
        if (elapsed == 0L) {
            return this.bytesRead;
        }
        return this.bytesRead / elapsed;
    }

    public long getTotalSleepTime() {
        return this.totalSleepTime;
    }

    public String toString() {
        return "ThrottledInputStream{bytesRead=" + this.bytesRead + ", maxBytesPerSec=" + this.maxBytesPerSec + ", bytesPerSec=" + this.getBytesPerSec() + ", totalSleepTime=" + this.totalSleepTime + '}';
    }
}

