/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import org.eclipse.jetty.http.ByteRange;
import org.eclipse.jetty.http.CompressedContentFormat;
import org.eclipse.jetty.http.DateGenerator;
import org.eclipse.jetty.http.EtagUtils;
import org.eclipse.jetty.http.HttpDateTime;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpFields;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpURI;
import org.eclipse.jetty.http.MultiPart;
import org.eclipse.jetty.http.MultiPartByteRanges;
import org.eclipse.jetty.http.PreEncodedHttpField;
import org.eclipse.jetty.http.QuotedCSV;
import org.eclipse.jetty.http.QuotedQualityCSV;
import org.eclipse.jetty.http.content.HttpContent;
import org.eclipse.jetty.http.content.PreCompressedHttpContent;
import org.eclipse.jetty.io.Content;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.ResourceListing;
import org.eclipse.jetty.server.Response;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.TypeUtil;
import org.eclipse.jetty.util.URIUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResourceService {
    private static final Logger LOG = LoggerFactory.getLogger(ResourceService.class);
    private static final int NO_CONTENT_LENGTH = -1;
    private static final int USE_KNOWN_CONTENT_LENGTH = -2;
    private static final EnumSet<HttpHeader> CONTENT_HEADERS = EnumSet.of(HttpHeader.LAST_MODIFIED, HttpHeader.CONTENT_LENGTH, HttpHeader.CONTENT_TYPE);
    private static final PreEncodedHttpField ACCEPT_RANGES_BYTES = new PreEncodedHttpField(HttpHeader.ACCEPT_RANGES, "bytes");
    private final List<CompressedContentFormat> _precompressedFormats = new ArrayList<CompressedContentFormat>();
    private final Map<String, List<String>> _preferredEncodingOrderCache = new ConcurrentHashMap<String, List<String>>();
    private final List<String> _preferredEncodingOrder = new ArrayList<String>();
    private WelcomeFactory _welcomeFactory;
    private WelcomeMode _welcomeMode = WelcomeMode.SERVE;
    private boolean _etags = false;
    private List<String> _gzipEquivalentFileExtensions;
    private HttpContent.Factory _contentFactory;
    private int _encodingCacheSize = 100;
    private boolean _dirAllowed = true;
    private boolean _acceptRanges = true;
    private HttpField _cacheControl;

    public HttpContent getContent(String path, Request request) throws IOException {
        HttpContent content = this._contentFactory.getContent(path == null ? "" : path);
        if (content != null) {
            List<String> preferredEncodingOrder;
            Collection<CompressedContentFormat> compressedContentFormats;
            ContextHandler aliasCheck = ContextHandler.getContextHandler(request);
            if (aliasCheck != null && !aliasCheck.checkAlias(path, content.getResource())) {
                return null;
            }
            Collection<CompressedContentFormat> collection = compressedContentFormats = content.getPreCompressedContentFormats() == null ? this._precompressedFormats : content.getPreCompressedContentFormats();
            if (!compressedContentFormats.isEmpty() && !(preferredEncodingOrder = this.getPreferredEncodingOrder(request)).isEmpty()) {
                for (String encoding : preferredEncodingOrder) {
                    HttpContent preCompressedContent;
                    CompressedContentFormat contentFormat = this.isEncodingAvailable(encoding, compressedContentFormats);
                    if (contentFormat == null || (preCompressedContent = this._contentFactory.getContent(path + contentFormat.getExtension())) == null || aliasCheck != null && !aliasCheck.checkAlias(path, preCompressedContent.getResource())) continue;
                    return new PreCompressedHttpContent(content, preCompressedContent, contentFormat);
                }
            }
        }
        return content;
    }

    public HttpContent.Factory getHttpContentFactory() {
        return this._contentFactory;
    }

    public void setHttpContentFactory(HttpContent.Factory contentFactory) {
        this._contentFactory = contentFactory;
    }

    public String getCacheControl() {
        return this._cacheControl.getValue();
    }

    public List<String> getGzipEquivalentFileExtensions() {
        return this._gzipEquivalentFileExtensions;
    }

    public void doGet(Request request, Response response, Callback callback, HttpContent content) {
        Object pathInContext = Request.getPathInContext(request);
        List<String> reqRanges = request.getHeaders().getValuesList(HttpHeader.RANGE.asString());
        if (!this._acceptRanges && !reqRanges.isEmpty()) {
            reqRanges = List.of();
            response.getHeaders().add(HttpHeader.ACCEPT_RANGES.asString(), "none");
        }
        boolean endsWithSlash = ((String)pathInContext).endsWith("/");
        if (LOG.isDebugEnabled()) {
            LOG.debug(".doGet(req={}, resp={}, callback={}, content={}) pathInContext={}, reqRanges={}, endsWithSlash={}", request, response, callback, content, pathInContext, reqRanges, endsWithSlash);
        }
        try {
            HttpField contentEncoding;
            if (content.getResource().isDirectory()) {
                this.sendWelcome(content, (String)pathInContext, endsWithSlash, request, response, callback);
                return;
            }
            if (endsWithSlash && ((String)pathInContext).length() > 1) {
                String q = request.getHttpURI().getQuery();
                pathInContext = ((String)pathInContext).substring(0, ((String)pathInContext).length() - 1);
                if (q != null && q.length() != 0) {
                    pathInContext = (String)pathInContext + "?" + q;
                }
                this.sendRedirect(request, response, callback, URIUtil.addPaths(request.getContext().getContextPath(), (String)pathInContext));
                return;
            }
            if (this.handleConditionalHeaders(request, response, content, callback)) {
                return;
            }
            if (content.getPreCompressedContentFormats() == null || !content.getPreCompressedContentFormats().isEmpty()) {
                response.getHeaders().put(HttpHeader.VARY, HttpHeader.ACCEPT_ENCODING.asString());
            }
            if ((contentEncoding = content.getContentEncoding()) != null) {
                response.getHeaders().put(contentEncoding);
            } else if (this.isImplicitlyGzippedContent((String)pathInContext)) {
                response.getHeaders().put(HttpHeader.CONTENT_ENCODING, "gzip");
            }
            this.sendData(request, response, callback, content, reqRanges);
        }
        catch (Throwable t) {
            LOG.warn("Failed to serve resource: {}", pathInContext, (Object)t);
            if (!response.isCommitted()) {
                this.writeHttpError(request, response, callback, t);
            }
            callback.failed(t);
        }
    }

    protected void writeHttpError(Request request, Response response, Callback callback, int status) {
        Response.writeError(request, response, callback, status);
    }

    protected void writeHttpError(Request request, Response response, Callback callback, Throwable cause) {
        Response.writeError(request, response, callback, cause);
    }

    protected void writeHttpError(Request request, Response response, Callback callback, int status, String msg, Throwable cause) {
        Response.writeError(request, response, callback, status, msg, cause);
    }

    protected void sendRedirect(Request request, Response response, Callback callback, String target) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("sendRedirect(req={}, resp={}, callback={}, target={})", request, response, callback, target);
        }
        Response.sendRedirect(request, response, callback, target);
    }

    private List<String> getPreferredEncodingOrder(Request request) {
        List<String> values;
        Enumeration<String> headers = request.getHeaders().getValues(HttpHeader.ACCEPT_ENCODING.asString());
        if (!headers.hasMoreElements()) {
            return Collections.emptyList();
        }
        String key = headers.nextElement();
        if (headers.hasMoreElements()) {
            StringBuilder sb = new StringBuilder(key.length() * 2);
            do {
                sb.append(',').append(headers.nextElement());
            } while (headers.hasMoreElements());
            key = sb.toString();
        }
        if ((values = this._preferredEncodingOrderCache.get(key)) == null) {
            QuotedQualityCSV encodingQualityCSV = new QuotedQualityCSV(this._preferredEncodingOrder);
            encodingQualityCSV.addValue(key);
            values = encodingQualityCSV.getValues();
            if (this._preferredEncodingOrderCache.size() > this._encodingCacheSize) {
                this._preferredEncodingOrderCache.clear();
            }
            this._preferredEncodingOrderCache.put(key, values);
        }
        return values;
    }

    private boolean isImplicitlyGzippedContent(String path) {
        if (path == null || this._gzipEquivalentFileExtensions == null) {
            return false;
        }
        for (String suffix : this._gzipEquivalentFileExtensions) {
            if (!path.endsWith(suffix)) continue;
            return true;
        }
        return false;
    }

    private CompressedContentFormat isEncodingAvailable(String encoding, Collection<CompressedContentFormat> availableFormats) {
        if (availableFormats.isEmpty()) {
            return null;
        }
        for (CompressedContentFormat format : availableFormats) {
            if (!format.getEncoding().equals(encoding)) continue;
            return format;
        }
        if ("*".equals(encoding)) {
            return availableFormats.iterator().next();
        }
        return null;
    }

    @Deprecated(forRemoval=true, since="12.1.0")
    protected boolean passConditionalHeaders(Request request, Response response, HttpContent content, Callback callback) throws IOException {
        return this.handleConditionalHeaders(request, response, content, callback);
    }

    protected boolean handleConditionalHeaders(Request request, Response response, HttpContent content, Callback callback) throws IOException {
        try {
            long lm;
            long ifumsl;
            String etag;
            String ifm = null;
            String ifnm = null;
            String ifms = null;
            String ifums = null;
            for (HttpField field : request.getHeaders()) {
                if (field.getHeader() == null) continue;
                switch (field.getHeader()) {
                    case IF_MATCH: {
                        ifm = field.getValue();
                        break;
                    }
                    case IF_NONE_MATCH: {
                        ifnm = field.getValue();
                        break;
                    }
                    case IF_MODIFIED_SINCE: {
                        ifms = field.getValue();
                        break;
                    }
                    case IF_UNMODIFIED_SINCE: {
                        ifums = field.getValue();
                        break;
                    }
                }
            }
            if (this._etags && (etag = content.getETagValue()) != null) {
                String matched;
                etag = EtagUtils.rewriteWithSuffix(content.getETagValue(), "");
                if (ifm != null && (matched = this.matchesEtag(etag, ifm)) == null) {
                    this.writeHttpError(request, response, callback, 412);
                    return true;
                }
                if (ifnm != null) {
                    matched = this.matchesEtag(etag, ifnm);
                    if (matched != null) {
                        response.getHeaders().put(HttpHeader.ETAG, matched);
                        this.putNotModifiedHeaders(response, content);
                        this.writeHttpError(request, response, callback, 304);
                        return true;
                    }
                    return false;
                }
            }
            if (ifms != null && ifnm == null) {
                long lm2;
                String mdlm = DateGenerator.formatDate(content.getLastModifiedInstant());
                if (ifms.equals(mdlm)) {
                    this.putNotModifiedHeaders(response, content);
                    this.writeHttpError(request, response, callback, 304);
                    return true;
                }
                long ifmsl = HttpDateTime.parseToEpoch(ifms);
                if (ifmsl != -1L && (lm2 = content.getResource().lastModified().toEpochMilli()) != -1L && lm2 / 1000L <= ifmsl / 1000L) {
                    this.putNotModifiedHeaders(response, content);
                    this.writeHttpError(request, response, callback, 304);
                    return true;
                }
            }
            if (ifums != null && ifm == null && (ifumsl = HttpDateTime.parseToEpoch(ifums)) != -1L && (lm = content.getResource().lastModified().toEpochMilli()) != -1L && lm / 1000L > ifumsl / 1000L) {
                this.writeHttpError(request, response, callback, 412);
                return true;
            }
        }
        catch (IllegalArgumentException iae) {
            if (!response.isCommitted()) {
                this.writeHttpError(request, response, callback, 400, null, iae);
            }
            throw iae;
        }
        return false;
    }

    private String matchesEtag(String contentETag, String requestEtag) {
        if (contentETag == null || requestEtag == null) {
            return null;
        }
        QuotedCSV quoted = new QuotedCSV(true, requestEtag);
        for (String tag : quoted) {
            if (!EtagUtils.matches(contentETag, tag)) continue;
            return tag;
        }
        return null;
    }

    protected void sendWelcome(HttpContent content, String pathInContext, boolean endsWithSlash, Request request, Response response, Callback callback) throws Exception {
        HttpURI.Mutable uri;
        if (!Objects.requireNonNull(content).getResource().isDirectory()) {
            throw new IllegalArgumentException("content must be a directory");
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("sendWelcome(content={}, pathInContext={}, endsWithSlash={}, req={}, resp={}, callback={})", content, pathInContext, endsWithSlash, request, response, callback);
        }
        if (!endsWithSlash && !(uri = HttpURI.build(request.getHttpURI())).getCanonicalPath().endsWith("/")) {
            uri.path(uri.getCanonicalPath() + "/");
            response.getHeaders().put(HttpFields.CONTENT_LENGTH_0);
            this.sendRedirect(request, response, callback, uri.getPathQuery());
            return;
        }
        if (this.welcome(content, request, response, callback)) {
            return;
        }
        if (!this.handleConditionalHeaders(request, response, content, callback)) {
            this.sendDirectory(request, response, content, callback, pathInContext);
        }
    }

    private boolean welcome(HttpContent content, Request request, Response response, Callback callback) throws Exception {
        WelcomeAction welcomeAction = this.processWelcome(content, request);
        if (LOG.isDebugEnabled()) {
            LOG.debug("welcome(req={}, rsp={}, cbk={}) welcomeAction={}", request, response, callback, welcomeAction);
        }
        if (welcomeAction == null) {
            return false;
        }
        this.handleWelcomeAction(request, response, callback, welcomeAction);
        return true;
    }

    protected void handleWelcomeAction(Request request, Response response, Callback callback, WelcomeAction welcomeAction) throws Exception {
        switch (welcomeAction.mode.ordinal()) {
            case 0: {
                this.redirectWelcome(request, response, callback, welcomeAction.target);
                break;
            }
            case 1: {
                this.serveWelcome(request, response, callback, welcomeAction.target);
                break;
            }
            case 2: {
                this.rehandleWelcome(request, response, callback, welcomeAction.target);
            }
        }
    }

    protected void redirectWelcome(Request request, Response response, Callback callback, String welcomeTarget) throws Exception {
        response.getHeaders().put(HttpFields.CONTENT_LENGTH_0);
        this.sendRedirect(request, response, callback, welcomeTarget);
    }

    protected void serveWelcome(Request request, Response response, Callback callback, String welcomeTarget) throws Exception {
        HttpContent httpContent = this._contentFactory.getContent(welcomeTarget);
        if (this.handleConditionalHeaders(request, response, httpContent, callback)) {
            return;
        }
        this.sendData(request, response, callback, httpContent, List.of());
    }

    protected void rehandleWelcome(Request request, Response response, Callback callback, String welcomeTarget) throws Exception {
        Response.writeError(request, response, callback, 500);
    }

    private WelcomeAction processWelcome(HttpContent content, Request request) throws IOException {
        String welcomeTarget = this.getWelcomeFactory().getWelcomeTarget(content, request);
        if (welcomeTarget == null) {
            return null;
        }
        String contextPath = request.getContext().getContextPath();
        WelcomeMode welcomeMode = this.getWelcomeMode();
        switch (welcomeMode.ordinal()) {
            default: {
                throw new IncompatibleClassChangeError();
            }
            case 0: 
            case 2: {
                String string = HttpURI.build(request.getHttpURI()).path(URIUtil.addPaths(contextPath, welcomeTarget)).getPathQuery();
                break;
            }
            case 1: {
                String string = welcomeTarget = welcomeTarget;
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("welcome {} {}", (Object)welcomeMode, (Object)welcomeTarget);
        }
        return new WelcomeAction(welcomeTarget, welcomeMode);
    }

    private void sendDirectory(Request request, Response response, HttpContent httpContent, Callback callback, String pathInContext) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("sendDirectory(req={}, resp={}, content={}, callback={}, pathInContext={})", request, response, httpContent, callback, pathInContext);
        }
        if (!this._dirAllowed) {
            this.writeHttpError(request, response, callback, 403);
            return;
        }
        String base = URIUtil.addEncodedPaths(request.getHttpURI().getPath(), "/");
        String listing = ResourceListing.getAsXHTML(httpContent.getResource(), base, pathInContext.length() > 1, request.getHttpURI().getQuery());
        if (listing == null) {
            this.writeHttpError(request, response, callback, 403);
            return;
        }
        String characterEncoding = httpContent.getCharacterEncoding();
        Charset charset = characterEncoding == null ? StandardCharsets.UTF_8 : Charset.forName(characterEncoding);
        byte[] data = listing.getBytes(charset);
        response.getHeaders().put(HttpHeader.CONTENT_TYPE, "text/html;charset=" + charset.name());
        response.getHeaders().put(HttpHeader.CONTENT_LENGTH, (long)data.length);
        response.write(true, ByteBuffer.wrap(data), callback);
    }

    private void sendData(Request request, Response response, Callback callback, HttpContent content, List<String> reqRanges) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("sendData(req={}, resp={}, callback={}) content={}, reqRanges={})", request, response, callback, content, reqRanges);
        }
        long contentLength = content.getContentLengthValue();
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("sendData content=%s", content));
        }
        if (reqRanges.isEmpty()) {
            if (contentLength >= 0L) {
                this.putHeaders(response, content, -2L);
            } else {
                this.putHeaders(response, content, -1L);
            }
            content.writeTo(response, 0L, -1L, callback);
            return;
        }
        List<ByteRange> ranges = ByteRange.parse(reqRanges, contentLength);
        if (ranges.isEmpty()) {
            response.getHeaders().put(HttpHeader.CONTENT_RANGE, ByteRange.toNonSatisfiableHeaderValue(contentLength));
            Response.writeError(request, response, callback, 416);
            return;
        }
        if (ranges.size() == 1) {
            ByteRange range2 = ranges.get(0);
            this.putHeaders(response, content, range2.getLength());
            response.setStatus(206);
            response.getHeaders().put(HttpHeader.CONTENT_RANGE, range2.toHeaderValue(contentLength));
            content.writeTo(response, range2.first(), range2.getLength(), callback);
            return;
        }
        response.setStatus(206);
        String contentType = "multipart/byteranges; boundary=";
        String boundary = MultiPart.generateBoundary(null, 24);
        MultiPartByteRanges.ContentSource byteRanges = new MultiPartByteRanges.ContentSource(boundary);
        ranges.forEach(range -> byteRanges.addPart(new MultiPartByteRanges.Part(content.getContentTypeValue(), content.getResource(), (ByteRange)range, contentLength, request.getComponents().getByteBufferPool())));
        byteRanges.close();
        long partsContentLength = byteRanges.getLength();
        this.putHeaders(response, content, partsContentLength);
        response.getHeaders().put(HttpHeader.CONTENT_TYPE, contentType + boundary);
        Content.copy(byteRanges, response, callback);
    }

    protected void putHeaders(Response response, HttpContent content, long contentLength) {
        HttpField ce;
        HttpField et;
        HttpFields.Mutable headers = response.getHeaders();
        if (this._etags && !headers.contains(HttpHeader.ETAG) && (et = content.getETag()) != null) {
            headers.add(et);
        }
        if ((ce = content.getContentEncoding()) != null) {
            headers.put(ce);
        }
        headers.remove(CONTENT_HEADERS);
        HttpField lm = content.getLastModified();
        if (lm != null) {
            headers.add(lm);
        }
        if (contentLength == -2L) {
            headers.add(content.getContentLength());
        } else if (contentLength > -1L) {
            headers.add(HttpHeader.CONTENT_LENGTH, contentLength);
        }
        HttpField ct = content.getContentType();
        if (ct != null) {
            headers.add(ct);
        }
        this.putHeaders(response);
    }

    protected void putNotModifiedHeaders(Response response, HttpContent content) {
        HttpFields.Mutable headers = response.getHeaders();
        HttpField lm = content.getLastModified();
        if (lm != null) {
            headers.put(lm);
        }
        this.putHeaders(response);
    }

    protected void putHeaders(Response response) {
        HttpFields.Mutable headers = response.getHeaders();
        if (this._acceptRanges && !headers.contains(HttpHeader.ACCEPT_RANGES)) {
            headers.add(ACCEPT_RANGES_BYTES);
        }
        if (this._cacheControl != null && !headers.contains(HttpHeader.CACHE_CONTROL)) {
            headers.add(this._cacheControl);
        }
    }

    public boolean isAcceptRanges() {
        return this._acceptRanges;
    }

    public boolean isDirAllowed() {
        return this._dirAllowed;
    }

    public boolean isEtags() {
        return this._etags;
    }

    public List<CompressedContentFormat> getPrecompressedFormats() {
        return this._precompressedFormats;
    }

    public WelcomeMode getWelcomeMode() {
        return this._welcomeMode;
    }

    public WelcomeFactory getWelcomeFactory() {
        return this._welcomeFactory;
    }

    public void setAcceptRanges(boolean acceptRanges) {
        this._acceptRanges = acceptRanges;
    }

    public void setCacheControl(String cacheControl) {
        this._cacheControl = new PreEncodedHttpField(HttpHeader.CACHE_CONTROL, cacheControl);
    }

    public void setDirAllowed(boolean dirAllowed) {
        this._dirAllowed = dirAllowed;
    }

    public void setEtags(boolean etags) {
        this._etags = etags;
    }

    public void setGzipEquivalentFileExtensions(List<String> gzipEquivalentFileExtensions) {
        this._gzipEquivalentFileExtensions = gzipEquivalentFileExtensions;
    }

    public void setPrecompressedFormats(List<CompressedContentFormat> precompressedFormats) {
        this._precompressedFormats.clear();
        this._precompressedFormats.addAll(precompressedFormats);
        this._preferredEncodingOrder.clear();
        this._preferredEncodingOrder.addAll(this._precompressedFormats.stream().map(CompressedContentFormat::getEncoding).toList());
    }

    public void setEncodingCacheSize(int encodingCacheSize) {
        this._encodingCacheSize = encodingCacheSize;
        if (encodingCacheSize > this._preferredEncodingOrderCache.size()) {
            this._preferredEncodingOrderCache.clear();
        }
    }

    public int getEncodingCacheSize() {
        return this._encodingCacheSize;
    }

    public void setWelcomeMode(WelcomeMode welcomeMode) {
        this._welcomeMode = Objects.requireNonNull(welcomeMode);
    }

    public String toString() {
        return String.format("%s@%x(contentFactory=%s, dirAllowed=%b, welcomeMode=%s)", new Object[]{TypeUtil.toShortName(this.getClass()), this.hashCode(), this._contentFactory, this._dirAllowed, this._welcomeMode});
    }

    public void setWelcomeFactory(WelcomeFactory welcomeFactory) {
        this._welcomeFactory = welcomeFactory;
    }

    public static enum WelcomeMode {
        REDIRECT,
        SERVE,
        REHANDLE;

    }

    public record WelcomeAction(String target, WelcomeMode mode) {
    }

    public static interface WelcomeFactory {
        public String getWelcomeTarget(HttpContent var1, Request var2) throws IOException;
    }
}

