/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.ratis;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.hdds.utils.DBStoreHAManager;
import org.apache.hadoop.hdds.utils.NettyMetrics;
import org.apache.hadoop.hdds.utils.TransactionInfo;
import org.apache.hadoop.ozone.audit.AuditLogger;
import org.apache.hadoop.ozone.audit.AuditLoggerType;
import org.apache.hadoop.ozone.audit.OMSystemAction;
import org.apache.hadoop.ozone.om.OzoneManager;
import org.apache.hadoop.ozone.om.OzoneManagerPrepareState;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.execution.flowcontrol.ExecutionContext;
import org.apache.hadoop.ozone.om.helpers.OMRatisHelper;
import org.apache.hadoop.ozone.om.lock.OMLockDetails;
import org.apache.hadoop.ozone.om.ratis.OzoneManagerDoubleBuffer;
import org.apache.hadoop.ozone.om.ratis.OzoneManagerRatisServer;
import org.apache.hadoop.ozone.om.ratis.utils.OzoneManagerRatisUtils;
import org.apache.hadoop.ozone.om.response.DummyOMClientResponse;
import org.apache.hadoop.ozone.om.response.OMClientResponse;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.ozone.protocolPB.OzoneManagerRequestHandler;
import org.apache.hadoop.ozone.protocolPB.RequestHandler;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftClientRequest;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeer;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.exceptions.StateMachineException;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.BaseStateMachine;
import org.apache.ratis.statemachine.impl.SimpleStateMachineStorage;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.ExitUtils;
import org.apache.ratis.util.IOUtils;
import org.apache.ratis.util.LifeCycle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OzoneManagerStateMachine
extends BaseStateMachine {
    private static final Logger LOG = LoggerFactory.getLogger(OzoneManagerStateMachine.class);
    private static final AuditLogger AUDIT = new AuditLogger(AuditLoggerType.OMSYSTEMLOGGER);
    private static final String AUDIT_PARAM_PREVIOUS_LEADER = "previousLeader";
    private static final String AUDIT_PARAM_NEW_LEADER = "newLeader";
    private RaftPeerId previousLeaderId = null;
    private final SimpleStateMachineStorage storage = new SimpleStateMachineStorage();
    private final OzoneManager ozoneManager;
    private RequestHandler handler;
    private volatile OzoneManagerDoubleBuffer ozoneManagerDoubleBuffer;
    private final ExecutorService executorService;
    private final ExecutorService installSnapshotExecutor;
    private final boolean isTracingEnabled;
    private final AtomicInteger statePausedCount = new AtomicInteger(0);
    private final String threadPrefix;
    private volatile TermIndex lastNotifiedTermIndex = TermIndex.valueOf((long)0L, (long)-1L);
    private volatile long lastSkippedIndex = -1L;
    private final NettyMetrics nettyMetrics;

    public OzoneManagerStateMachine(OzoneManagerRatisServer ratisServer, boolean isTracingEnabled) throws IOException {
        this.isTracingEnabled = isTracingEnabled;
        this.ozoneManager = ratisServer.getOzoneManager();
        this.loadSnapshotInfoFromDB();
        this.threadPrefix = this.ozoneManager.getThreadNamePrefix();
        this.ozoneManagerDoubleBuffer = this.buildDoubleBufferForRatis();
        this.handler = new OzoneManagerRequestHandler(this.ozoneManager);
        ThreadFactory build = new ThreadFactoryBuilder().setDaemon(true).setNameFormat(String.valueOf(this.threadPrefix) + "OMStateMachineApplyTransactionThread - %d").build();
        this.executorService = HadoopExecutors.newSingleThreadExecutor((ThreadFactory)build);
        ThreadFactory installSnapshotThreadFactory = new ThreadFactoryBuilder().setNameFormat(String.valueOf(this.threadPrefix) + "InstallSnapshotThread").build();
        this.installSnapshotExecutor = HadoopExecutors.newSingleThreadExecutor((ThreadFactory)installSnapshotThreadFactory);
        this.nettyMetrics = NettyMetrics.create();
    }

    public void initialize(RaftServer server, RaftGroupId id, RaftStorage raftStorage) throws IOException {
        this.getLifeCycle().startAndTransition(() -> {
            super.initialize(server, id, raftStorage);
            this.storage.init(raftStorage);
            LOG.info("{}: initialize {} with {}", new Object[]{this.getId(), id, this.getLastAppliedTermIndex()});
        }, new Class[0]);
    }

    public synchronized void reinitialize() throws IOException {
        this.loadSnapshotInfoFromDB();
        if (this.getLifeCycleState() == LifeCycle.State.PAUSED) {
            TermIndex lastApplied = this.getLastAppliedTermIndex();
            this.unpause(lastApplied.getIndex(), lastApplied.getTerm());
            LOG.info("{}: reinitialize {} with {}", new Object[]{this.getId(), this.getGroupId(), lastApplied});
        }
    }

    public SnapshotInfo getLatestSnapshot() {
        SnapshotInfo snapshotInfo = this.ozoneManager.getTransactionInfo().toSnapshotInfo();
        LOG.debug("Latest Snapshot Info {}", (Object)snapshotInfo);
        return snapshotInfo;
    }

    public void notifyLeaderReady() {
        this.ozoneManager.getOmSnapshotManager().resetInFlightSnapshotCount();
    }

    public void notifyLeaderChanged(RaftGroupMemberId groupMemberId, RaftPeerId newLeaderId) {
        RaftPeerId currentPeerId = groupMemberId.getPeerId();
        if (newLeaderId.equals((Object)currentPeerId)) {
            this.ozoneManager.initializeEdekCache(this.ozoneManager.getConfiguration());
        }
        RaftPeerId actualPreviousLeader = this.previousLeaderId;
        this.previousLeaderId = newLeaderId;
        this.ozoneManager.omHAMetricsInit(newLeaderId.toString());
        LinkedHashMap<String, String> auditParams = new LinkedHashMap<String, String>();
        auditParams.put(AUDIT_PARAM_PREVIOUS_LEADER, actualPreviousLeader != null ? String.valueOf(actualPreviousLeader) : "NONE");
        auditParams.put(AUDIT_PARAM_NEW_LEADER, String.valueOf(newLeaderId));
        AUDIT.logWriteSuccess(this.ozoneManager.buildAuditMessageForSuccess(OMSystemAction.LEADER_CHANGE, auditParams));
        LOG.info("{}: leader changed to {}", (Object)groupMemberId, (Object)newLeaderId);
    }

    public synchronized void notifyTermIndexUpdated(long currentTerm, long newIndex) {
        long oldIndex = this.lastNotifiedTermIndex.getIndex();
        if (newIndex - oldIndex > 1L) {
            this.lastSkippedIndex = newIndex - 1L;
        }
        TermIndex newTermIndex = TermIndex.valueOf((long)currentTerm, (long)newIndex);
        this.lastNotifiedTermIndex = this.assertUpdateIncreasingly("lastNotified", this.lastNotifiedTermIndex, newTermIndex);
        if (this.lastNotifiedTermIndex.getIndex() - this.getLastAppliedTermIndex().getIndex() == 1L) {
            this.updateLastAppliedTermIndex(this.lastNotifiedTermIndex);
        }
    }

    public TermIndex getLastNotifiedTermIndex() {
        return this.lastNotifiedTermIndex;
    }

    protected synchronized boolean updateLastAppliedTermIndex(TermIndex newTermIndex) {
        TermIndex lastApplied = this.getLastAppliedTermIndex();
        this.assertUpdateIncreasingly("lastApplied", lastApplied, newTermIndex);
        if (newTermIndex.getIndex() < this.getLastNotifiedTermIndex().getIndex() && newTermIndex.getIndex() >= this.lastSkippedIndex) {
            newTermIndex = this.getLastNotifiedTermIndex();
        }
        return super.updateLastAppliedTermIndex(newTermIndex);
    }

    private TermIndex assertUpdateIncreasingly(String name, TermIndex oldTermIndex, TermIndex newTermIndex) {
        Preconditions.checkArgument((newTermIndex.compareTo(oldTermIndex) >= 0 ? 1 : 0) != 0, (String)"%s: newTermIndex = %s < oldTermIndex = %s", (Object)name, (Object)newTermIndex, (Object)oldTermIndex);
        return newTermIndex;
    }

    public void notifyConfigurationChanged(long term, long index, RaftProtos.RaftConfigurationProto newRaftConfiguration) {
        List newPeers = newRaftConfiguration.getPeersList();
        List newListeners = newRaftConfiguration.getListenersList();
        StringBuilder logBuilder = new StringBuilder(1024).append("notifyConfigurationChanged from Ratis: term=").append(term).append(", index=").append(index).append(", New Peer list: ");
        newPeers.forEach(peer -> {
            StringBuilder stringBuilder2 = logBuilder.append(peer.getId().toStringUtf8()).append('(').append(peer.getAddress()).append("), ");
        });
        logBuilder.append("New Listener list: ");
        newListeners.forEach(peer -> {
            StringBuilder stringBuilder2 = logBuilder.append(peer.getId().toStringUtf8()).append('(').append(peer.getAddress()).append("), ");
        });
        LOG.info(logBuilder.substring(0, logBuilder.length() - 2));
        ArrayList<String> newPeerIds = new ArrayList<String>();
        for (RaftProtos.RaftPeerProto raftPeerProto : newPeers) {
            newPeerIds.add(RaftPeerId.valueOf((ByteString)raftPeerProto.getId()).toString());
        }
        for (RaftProtos.RaftPeerProto raftPeerProto : newListeners) {
            newPeerIds.add(RaftPeerId.valueOf((ByteString)raftPeerProto.getId()).toString());
        }
        this.ozoneManager.updatePeerList(newPeerIds);
    }

    public void notifySnapshotInstalled(RaftProtos.InstallSnapshotResult result, long snapshotIndex, RaftPeer peer) {
        LOG.info("Receive notifySnapshotInstalled event {} for the peer: {} snapshotIndex: {}.", new Object[]{result, peer.getId(), snapshotIndex});
        switch (result) {
            case SUCCESS: 
            case SNAPSHOT_UNAVAILABLE: {
                if (!this.ozoneManager.getOmRatisServer().getServerDivision().getPeer().equals((Object)peer)) break;
                this.ozoneManager.getOmSnapshotProvider().init();
                break;
            }
        }
    }

    public TransactionContext startTransaction(RaftClientRequest raftClientRequest) throws IOException {
        ByteString messageContent = raftClientRequest.getMessage().getContent();
        OzoneManagerProtocolProtos.OMRequest omRequest = OMRatisHelper.convertByteStringToOMRequest((ByteString)messageContent);
        Preconditions.checkArgument((boolean)raftClientRequest.getRaftGroupId().equals((Object)this.getGroupId()));
        try {
            this.handler.validateRequest(omRequest);
        }
        catch (IOException ioe) {
            TransactionContext ctxt = TransactionContext.newBuilder().setClientRequest(raftClientRequest).setStateMachine((StateMachine)this).setServerRole(RaftProtos.RaftPeerRole.LEADER).build();
            ctxt.setException((Exception)ioe);
            return ctxt;
        }
        return TransactionContext.newBuilder().setClientRequest(raftClientRequest).setStateMachine((StateMachine)this).setServerRole(RaftProtos.RaftPeerRole.LEADER).setLogData(raftClientRequest.getMessage().getContent()).setStateMachineContext((Object)omRequest).build();
    }

    public TransactionContext preAppendTransaction(TransactionContext trx) throws IOException {
        OzoneManagerProtocolProtos.OMRequest request = (OzoneManagerProtocolProtos.OMRequest)trx.getStateMachineContext();
        OzoneManagerProtocolProtos.Type cmdType = request.getCmdType();
        OzoneManagerPrepareState prepareState = this.ozoneManager.getPrepareState();
        if (LOG.isDebugEnabled()) {
            LOG.debug("{}: preAppendTransaction {}", (Object)this.getId(), (Object)TermIndex.valueOf((RaftProtos.LogEntryProto)trx.getLogEntry()));
        }
        if (cmdType == OzoneManagerProtocolProtos.Type.Prepare) {
            UserGroupInformation userGroupInformation = UserGroupInformation.createRemoteUser((String)request.getUserInfo().getUserName());
            if (this.ozoneManager.getAclsEnabled() && !this.ozoneManager.isAdmin(userGroupInformation)) {
                String message = "Access denied for user " + userGroupInformation + ". Superuser privilege is required to prepare upgrade/downgrade.";
                OMException cause = new OMException(message, OMException.ResultCodes.ACCESS_DENIED);
                throw new StateMachineException(message, (Throwable)cause, false);
            }
            prepareState.enablePrepareGate();
        }
        if (prepareState.requestAllowed(cmdType)) {
            return trx;
        }
        String message = "Cannot apply write request " + request.getCmdType().name() + " when OM is in prepare mode.";
        OMException cause = new OMException(message, OMException.ResultCodes.NOT_SUPPORTED_OPERATION_WHEN_PREPARED);
        throw new StateMachineException(message, (Throwable)cause, false);
    }

    public CompletableFuture<Message> applyTransaction(TransactionContext trx) {
        try {
            Object context = trx.getStateMachineContext();
            OzoneManagerProtocolProtos.OMRequest request = context != null ? (OzoneManagerProtocolProtos.OMRequest)context : OMRatisHelper.convertByteStringToOMRequest((ByteString)trx.getStateMachineLogEntry().getLogData());
            TermIndex termIndex = TermIndex.valueOf((RaftProtos.LogEntryProto)trx.getLogEntry());
            LOG.debug("{}: applyTransaction {}", (Object)this.getId(), (Object)termIndex);
            this.ozoneManagerDoubleBuffer.acquireUnFlushedTransactions(1);
            return CompletableFuture.supplyAsync(() -> this.runCommand(request, termIndex), this.executorService).thenApply(this::processResponse);
        }
        catch (Exception e) {
            return OzoneManagerStateMachine.completeExceptionally(e);
        }
    }

    private Message processResponse(OzoneManagerProtocolProtos.OMResponse omResponse) {
        if (!omResponse.getSuccess()) {
            if (omResponse.getStatus() == OzoneManagerProtocolProtos.Status.INTERNAL_ERROR) {
                OzoneManagerStateMachine.terminate(omResponse, OMException.ResultCodes.INTERNAL_ERROR);
            } else if (omResponse.getStatus() == OzoneManagerProtocolProtos.Status.METADATA_ERROR) {
                OzoneManagerStateMachine.terminate(omResponse, OMException.ResultCodes.METADATA_ERROR);
            }
        }
        return OMRatisHelper.convertResponseToMessage((OzoneManagerProtocolProtos.OMResponse)omResponse);
    }

    private static void terminate(OzoneManagerProtocolProtos.OMResponse omResponse, OMException.ResultCodes resultCode) {
        OMException exception = new OMException(omResponse.getMessage(), resultCode);
        String errorMessage = "OM Ratis Server has received unrecoverable error, to avoid further DB corruption, terminating OM. Error Response received is:" + omResponse;
        ExitUtils.terminate((int)1, (String)errorMessage, (Throwable)exception, (Logger)LOG);
    }

    public CompletableFuture<Message> query(Message request) {
        try {
            OzoneManagerProtocolProtos.OMRequest omRequest = OMRatisHelper.convertByteStringToOMRequest((ByteString)request.getContent());
            return CompletableFuture.completedFuture(this.queryCommand(omRequest));
        }
        catch (IOException e) {
            return OzoneManagerStateMachine.completeExceptionally(e);
        }
    }

    public synchronized void pause() {
        LOG.info("OzoneManagerStateMachine is pausing");
        this.statePausedCount.incrementAndGet();
        LifeCycle.State state = this.getLifeCycleState();
        if (state == LifeCycle.State.PAUSED) {
            return;
        }
        if (state != LifeCycle.State.NEW) {
            this.getLifeCycle().transition(LifeCycle.State.PAUSING);
            this.getLifeCycle().transition(LifeCycle.State.PAUSED);
        }
        this.ozoneManagerDoubleBuffer.stop();
    }

    public synchronized void unpause(long newLastAppliedSnaphsotIndex, long newLastAppliedSnapShotTermIndex) {
        if (this.statePausedCount.decrementAndGet() == 0) {
            this.getLifeCycle().startAndTransition(() -> {
                this.ozoneManagerDoubleBuffer = this.buildDoubleBufferForRatis();
                this.setLastAppliedTermIndex(TermIndex.valueOf((long)newLastAppliedSnapShotTermIndex, (long)newLastAppliedSnaphsotIndex));
                LOG.info("{}: OzoneManagerStateMachine un-pause completed. newLastAppliedSnapshotIndex: {}, newLastAppliedSnapShotTermIndex: {}", new Object[]{this.getId(), newLastAppliedSnaphsotIndex, newLastAppliedSnapShotTermIndex});
            }, new Class[0]);
        }
    }

    public OzoneManagerDoubleBuffer buildDoubleBufferForRatis() {
        int maxUnFlushedTransactionCount = this.ozoneManager.getConfiguration().getInt("ozone.om.unflushed.transaction.max.count", 10000);
        return OzoneManagerDoubleBuffer.newBuilder().setOmMetadataManager(this.ozoneManager.getMetadataManager()).setUpdateLastAppliedIndex(this::updateLastAppliedTermIndex).setMaxUnFlushedTransactionCount(maxUnFlushedTransactionCount).setThreadPrefix(this.threadPrefix).setS3SecretManager(this.ozoneManager.getS3SecretManager()).enableTracing(this.isTracingEnabled).build().start();
    }

    public long takeSnapshot() throws IOException {
        while (this.getLastAppliedTermIndex().getIndex() < this.lastSkippedIndex) {
            if (this.ozoneManager.isStopped()) {
                throw new IOException("OzoneManager is already stopped: " + this.ozoneManager.getNodeDetails());
            }
            try {
                this.ozoneManagerDoubleBuffer.awaitFlush();
            }
            catch (InterruptedException e) {
                throw IOUtils.toInterruptedIOException((String)"Interrupted ozoneManagerDoubleBuffer.awaitFlush", (InterruptedException)e);
            }
        }
        return this.takeSnapshotImpl();
    }

    private synchronized long takeSnapshotImpl() throws IOException {
        TermIndex notified;
        TermIndex applied = this.getLastAppliedTermIndex();
        TermIndex snapshot = applied.compareTo(notified = this.getLastNotifiedTermIndex()) > 0 ? applied : notified;
        long startTime = Time.monotonicNow();
        TransactionInfo transactionInfo = TransactionInfo.valueOf((TermIndex)snapshot);
        this.ozoneManager.setTransactionInfo(transactionInfo);
        this.ozoneManager.getMetadataManager().getTransactionInfoTable().put((Object)"#TRANSACTIONINFO", (Object)transactionInfo);
        this.ozoneManager.getMetadataManager().getStore().flushDB();
        LOG.info("{}: taking snapshot. applied = {}, skipped = {}, notified = {}, current snapshot index = {}, took {} ms", new Object[]{this.getId(), applied, this.lastSkippedIndex, notified, snapshot, Time.monotonicNow() - startTime});
        return snapshot.getIndex();
    }

    public CompletableFuture<TermIndex> notifyInstallSnapshotFromLeader(RaftProtos.RoleInfoProto roleInfoProto, TermIndex firstTermIndexInLog) {
        String leaderNodeId = RaftPeerId.valueOf((ByteString)roleInfoProto.getFollowerInfo().getLeaderInfo().getId().getId()).toString();
        LOG.info("Received install snapshot notification from OM leader: {} with term index: {}", (Object)leaderNodeId, (Object)firstTermIndexInLog);
        return CompletableFuture.supplyAsync(() -> this.ozoneManager.installSnapshotFromLeader(leaderNodeId), this.installSnapshotExecutor);
    }

    public String toStateMachineLogEntryString(RaftProtos.StateMachineLogEntryProto proto) {
        return OMRatisHelper.smProtoToString((RaftProtos.StateMachineLogEntryProto)proto);
    }

    public void close() {
        if (!this.ozoneManager.isStopped()) {
            LOG.info("Stopping {}. Shutdown also OzoneManager {}.", (Object)this, (Object)this.ozoneManager);
            this.ozoneManager.shutDown("OM state machine is shutdown by Ratis server");
        } else {
            LOG.info("Stopping {}.", (Object)this);
            this.stop();
        }
    }

    private OzoneManagerProtocolProtos.OMResponse runCommand(OzoneManagerProtocolProtos.OMRequest request, TermIndex termIndex) {
        try {
            ExecutionContext context = ExecutionContext.of(termIndex.getIndex(), termIndex);
            OMClientResponse omClientResponse = this.handler.handleWriteRequest(request, context, this.ozoneManagerDoubleBuffer);
            OMLockDetails omLockDetails = omClientResponse.getOmLockDetails();
            OzoneManagerProtocolProtos.OMResponse omResponse = omClientResponse.getOMResponse();
            if (omLockDetails != null) {
                return omResponse.toBuilder().setOmLockDetails(omLockDetails.toProtobufBuilder()).build();
            }
            return omResponse;
        }
        catch (IOException e) {
            LOG.warn("Failed to write, Exception occurred ", (Throwable)e);
            return this.createErrorResponse(request, e, termIndex);
        }
        catch (Throwable e) {
            String errorMessage = "Request " + request + " failed with exception";
            ExitUtils.terminate((int)1, (String)errorMessage, (Throwable)e, (Logger)LOG);
            return null;
        }
    }

    private OzoneManagerProtocolProtos.OMResponse createErrorResponse(OzoneManagerProtocolProtos.OMRequest omRequest, IOException exception, TermIndex termIndex) {
        OzoneManagerProtocolProtos.OMResponse.Builder omResponseBuilder = OzoneManagerProtocolProtos.OMResponse.newBuilder().setStatus(OzoneManagerRatisUtils.exceptionToResponseStatus(exception)).setCmdType(omRequest.getCmdType()).setTraceID(omRequest.getTraceID()).setSuccess(false);
        if (exception.getMessage() != null) {
            omResponseBuilder.setMessage(exception.getMessage());
        }
        OzoneManagerProtocolProtos.OMResponse omResponse = omResponseBuilder.build();
        DummyOMClientResponse omClientResponse = new DummyOMClientResponse(omResponse);
        this.ozoneManagerDoubleBuffer.add(omClientResponse, termIndex);
        return omResponse;
    }

    public void loadSnapshotInfoFromDB() throws IOException {
        TransactionInfo transactionInfo = TransactionInfo.readTransactionInfo((DBStoreHAManager)this.ozoneManager.getMetadataManager());
        if (transactionInfo != null) {
            TermIndex ti = transactionInfo.getTermIndex();
            this.setLastAppliedTermIndex(ti);
            this.ozoneManager.setTransactionInfo(transactionInfo);
            LOG.info("LastAppliedIndex is set from TransactionInfo from OM DB as {}", (Object)ti);
        } else {
            LOG.info("TransactionInfo not found in OM DB.");
        }
    }

    private Message queryCommand(OzoneManagerProtocolProtos.OMRequest request) {
        OzoneManagerProtocolProtos.OMResponse response = this.handler.handleReadRequest(request);
        return OMRatisHelper.convertResponseToMessage((OzoneManagerProtocolProtos.OMResponse)response);
    }

    private static <T> CompletableFuture<T> completeExceptionally(Exception e) {
        CompletableFuture future = new CompletableFuture();
        future.completeExceptionally(e);
        return future;
    }

    @VisibleForTesting
    public void setHandler(OzoneManagerRequestHandler handler) {
        this.handler = handler;
    }

    @VisibleForTesting
    public OzoneManagerRequestHandler getHandler() {
        return (OzoneManagerRequestHandler)this.handler;
    }

    public void stop() {
        this.ozoneManagerDoubleBuffer.stop();
        HadoopExecutors.shutdown((ExecutorService)this.executorService, (Logger)LOG, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        HadoopExecutors.shutdown((ExecutorService)this.installSnapshotExecutor, (Logger)LOG, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        if (this.nettyMetrics != null) {
            this.nettyMetrics.unregister();
        }
    }

    public void awaitDoubleBufferFlush() throws InterruptedException {
        this.ozoneManagerDoubleBuffer.awaitFlush();
    }

    @VisibleForTesting
    public OzoneManagerDoubleBuffer getOzoneManagerDoubleBuffer() {
        return this.ozoneManagerDoubleBuffer;
    }
}

