/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificate.utils;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.security.NoSuchProviderException;
import java.security.cert.CertPath;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertificateCodec {
    public static final String BEGIN_CERT = "-----BEGIN CERTIFICATE-----";
    public static final String END_CERT = "-----END CERTIFICATE-----";
    public static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;
    private static final Logger LOG = LoggerFactory.getLogger(CertificateCodec.class);
    private final SecurityConfig securityConfig;
    private final Path location;
    private final Set<PosixFilePermission> permissionSet = Stream.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE).collect(Collectors.toSet());

    public CertificateCodec(SecurityConfig config, String component) {
        this.securityConfig = config;
        this.location = this.securityConfig.getCertificateLocation(component);
    }

    public CertificateCodec(SecurityConfig config, Path certPath) {
        this.securityConfig = config;
        this.location = certPath;
    }

    public static String getPEMEncodedString(CertPath certPath) throws SCMSecurityException {
        List<? extends Certificate> certsInPath = certPath.getCertificates();
        ArrayList<String> pemEncodedList = new ArrayList<String>(certsInPath.size());
        for (Certificate certificate : certsInPath) {
            pemEncodedList.add(CertificateCodec.getPEMEncodedString((X509Certificate)certificate));
        }
        return StringUtils.join(pemEncodedList, (String)"\n");
    }

    public static <OUT extends OutputStream> OUT writePEMEncoded(X509Certificate certificate, OUT out) throws IOException {
        CertificateCodec.writePEMEncoded(certificate, new OutputStreamWriter(out, DEFAULT_CHARSET));
        return out;
    }

    public static <W extends Writer> W writePEMEncoded(X509Certificate certificate, W writer) throws IOException {
        try (JcaPEMWriter pemWriter = new JcaPEMWriter(writer);){
            pemWriter.writeObject((Object)certificate);
        }
        return writer;
    }

    public static String getPEMEncodedString(X509Certificate certificate) throws SCMSecurityException {
        try {
            return CertificateCodec.writePEMEncoded(certificate, new StringWriter()).toString();
        }
        catch (IOException e) {
            throw new SCMSecurityException("Failed to getPEMEncodedString for certificate with subject " + certificate.getSubjectDN(), e, SCMSecurityException.ErrorCode.PEM_ENCODE_FAILED);
        }
    }

    public static X509Certificate getX509Certificate(String pemEncoded) throws CertificateException {
        return CertificateCodec.getX509Certificate(pemEncoded.getBytes(DEFAULT_CHARSET));
    }

    public static X509Certificate getX509Certificate(byte[] pemEncoded) throws CertificateException {
        ByteArrayInputStream input = new ByteArrayInputStream(pemEncoded);
        return CertificateCodec.readX509Certificate(input);
    }

    public static X509Certificate readX509Certificate(InputStream input) throws CertificateException {
        Certificate cert = CertificateCodec.getCertFactory().generateCertificate(input);
        if (cert instanceof X509Certificate) {
            return (X509Certificate)cert;
        }
        throw new CertificateException("Certificate is not a X509Certificate: " + cert.getClass() + ", " + cert);
    }

    public static X509Certificate readX509Certificate(String pemEncoded) throws IOException {
        try {
            return CertificateCodec.getX509Certificate(pemEncoded);
        }
        catch (CertificateException e) {
            throw new IOException("Failed to getX509Certificate from " + pemEncoded, e);
        }
    }

    public static X509Certificate firstCertificateFrom(CertPath certificatePath) {
        return (X509Certificate)certificatePath.getCertificates().get(0);
    }

    public static CertificateFactory getCertFactory() throws CertificateException {
        try {
            return CertificateFactory.getInstance("X.509", "BC");
        }
        catch (NoSuchProviderException e) {
            throw new RuntimeException("BouncyCastle JCE provider not loaded.", e);
        }
    }

    public Path getLocation() {
        return this.location;
    }

    public void writeCertificate(X509Certificate xCertificate) throws IOException {
        String pem = CertificateCodec.getPEMEncodedString(xCertificate);
        this.writeCertificate(this.location.toAbsolutePath(), this.securityConfig.getCertificateFileName(), pem);
    }

    public void writeCertificate(X509Certificate xCertificate, String fileName) throws IOException {
        String pem = CertificateCodec.getPEMEncodedString(xCertificate);
        this.writeCertificate(this.location.toAbsolutePath(), fileName, pem);
    }

    public void writeCertificate(String fileName, String pemEncodedCert) throws IOException {
        this.writeCertificate(this.location.toAbsolutePath(), fileName, pemEncodedCert);
    }

    public synchronized void writeCertificate(Path basePath, String fileName, String pemEncodedCertificate) throws IOException {
        this.checkBasePathDirectory(basePath);
        File certificateFile = Paths.get(basePath.toString(), fileName).toFile();
        try (OutputStream file = Files.newOutputStream(certificateFile.toPath(), new OpenOption[0]);){
            file.write(pemEncodedCertificate.getBytes(DEFAULT_CHARSET));
        }
        LOG.info("Save certificate to {}", (Object)certificateFile.getAbsolutePath());
        LOG.info("Certificate {}", (Object)pemEncodedCertificate);
        Files.setPosixFilePermissions(certificateFile.toPath(), this.permissionSet);
    }

    public static CertPath getCertPathFromPemEncodedString(String pemString) throws CertificateException {
        return CertificateCodec.generateCertPathFromInputStream(new ByteArrayInputStream(pemString.getBytes(DEFAULT_CHARSET)));
    }

    private CertPath getCertPath(Path path, String fileName) throws IOException, CertificateException {
        this.checkBasePathDirectory(path.toAbsolutePath());
        File certFile = Paths.get(path.toAbsolutePath().toString(), fileName).toFile();
        if (!certFile.exists()) {
            throw new IOException("Unable to find the requested certificate file. Path: " + certFile);
        }
        try (InputStream is = Files.newInputStream(certFile.toPath(), new OpenOption[0]);){
            CertPath certPath = CertificateCodec.generateCertPathFromInputStream(is);
            return certPath;
        }
    }

    public CertPath getCertPath(String fileName) throws IOException, CertificateException {
        return this.getCertPath(this.location, fileName);
    }

    public CertPath getCertPath() throws CertificateException, IOException {
        return this.getCertPath(this.securityConfig.getCertificateFileName());
    }

    public CertPath prependCertToCertPath(X509Certificate certificate, CertPath path) throws CertificateException {
        List<? extends Certificate> certificates = path.getCertificates();
        ArrayList<X509Certificate> updatedList = new ArrayList<X509Certificate>();
        updatedList.add(certificate);
        for (Certificate certificate2 : certificates) {
            updatedList.add((X509Certificate)certificate2);
        }
        return CertificateCodec.getCertFactory().generateCertPath(updatedList);
    }

    public X509Certificate getTargetCert(Path path, String fileName) throws CertificateException, IOException {
        CertPath certPath = this.getCertPath(path, fileName);
        return CertificateCodec.firstCertificateFrom(certPath);
    }

    public X509Certificate getTargetCert() throws CertificateException, IOException {
        return this.getTargetCert(this.location, this.securityConfig.getCertificateFileName());
    }

    private static CertPath generateCertPathFromInputStream(InputStream inputStream) throws CertificateException {
        return CertificateCodec.getCertFactory().generateCertPath(inputStream, "PEM");
    }

    private void checkBasePathDirectory(Path basePath) throws IOException {
        if (!basePath.toFile().exists() && !basePath.toFile().mkdirs()) {
            LOG.error("Unable to create file path. Path: {}", (Object)basePath);
            throw new IOException("Creation of the directories failed." + basePath);
        }
    }
}

