/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.parquet.cli;

import java.io.File;
import java.io.FileWriter;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.util.ToolRunner;
import org.junit.Assert;
import org.junit.Test;
import org.slf4j.LoggerFactory;

public class MainTest {

  @Test
  public void mainTest() throws Exception {
    ToolRunner.run(new Configuration(), new Main(LoggerFactory.getLogger(MainTest.class)), new String[] {});
    Assert.assertTrue("we simply verify there are no errors here", true);
  }

  @Test
  public void testConfigFileLoading() throws Exception {
    File configFile = File.createTempFile("test-config", ".properties");
    configFile.deleteOnExit();

    try (FileWriter writer = new FileWriter(configFile)) {
      writer.write("test.key=test.value\n");
    }

    try {
      new Main(LoggerFactory.getLogger(MainTest.class))
          .run(new String[] {"--config-file", configFile.getAbsolutePath(), "help"});
      Assert.assertTrue("Config file loading should not throw exception", true);
    } catch (IllegalArgumentException e) {
      Assert.fail("Config file loading failed: " + e.getMessage());
    }
  }

  @Test
  public void testLocalPropertiesFile() throws Exception {
    String configFile = getClass().getResource("/test-config.properties").getPath();
    ToolRunner.run(new Configuration(), new Main(LoggerFactory.getLogger(MainTest.class)), new String[] {
      "--config-file", configFile, "version"
    });
  }

  @Test
  public void testLocalXmlFile() throws Exception {
    String configFile = getClass().getResource("/test-config.xml").getPath();
    ToolRunner.run(new Configuration(), new Main(LoggerFactory.getLogger(MainTest.class)), new String[] {
      "--config-file", configFile, "version"
    });
  }
}
